/*************************************************************************************************************
 *                                                                                                           *
 *  Copyright (c) 2015, Intel Corporation                                                                    *
 *                                                                                                           *
 *  Redistribution and use in source and binary forms, with or without                                       *
 *  modification, are permitted provided that the following conditions are met:                              *
 *                                                                                                           *
 *      * Redistributions of source code must retain the above copyright notice,                             *
 *        this list of conditions and the following disclaimer.                                              *
 *      * Redistributions in binary form must reproduce the above copyright                                  *
 *        notice, this list of conditions and the following disclaimer in the                                *
 *        documentation and/or other materials provided with the distribution.                               *
 *      * Neither the name of Intel Corporation nor the names of its contributors                            *
 *        may be used to endorse or promote products derived from this software                              *
 *        without specific prior written permission.                                                         *
 *                                                                                                           *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"                              *
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE                                *
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE                           *
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE                              *
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL                               *
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR                               *
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER                               *
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,                            *
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE                            *
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                     *
 *                                                                                                           *
 *************************************************************************************************************
 *                                                                                                           *
 *  Module name:                                                                                             *
 *      freebsddriveros_i.c                                                                                  *
 *                                                                                                           *
 *  Abstract:                                                                                                *
 *      This file contains freebsddriveros_i.c                                                               *
 *                                                                                                           *
 ************************************************************************************************************/

#include <naltypes.h>
#include <nalcodes.h>
#include <os_i.h>
#include <freebsdos_i.h>
#include <freebsddevice_i.h>
#include <sys/types.h>
#include <machine/atomic.h>

#include <sys/param.h>
#include <sys/module.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/uio.h>
#include <sys/types.h>
#include <sys/malloc.h>
#include <sys/param.h>
#include <sys/lock.h>
#include <sys/mutex.h>

#include <machine/bus.h>

#include <sys/bus.h>
#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>

#include <sys/param.h>
#include <vm/vm.h>
#include <vm/pmap.h>

MALLOC_DEFINE( M_NAL , "nal_device" , "nal driver allocations" );

NAL_FREEBSD_PCI_ALLOCATION_SLOT         Global_PciAllocationSlotsTable[NAL_FREEBSD_MAX_NON_PAGED_MEMORY_ALLOCATIONS];
NAL_FREEBSD_DMA_ALLOCATION_SLOT         Global_DmaAllocationSlotsTable[NAL_FREEBSD_MAX_NON_PAGED_MEMORY_ALLOCATIONS];
extern struct mtx                       Global_AtomicTestSetSpinLock;


VOID
_NalFreebsdDmaLoadCallback(
        IN  VOID *                              Argument,
        IN  bus_dma_segment_t *                 Segment,
        IN  int                                 NumberOfSegments,
        IN  int                                 Error
        );

NAL_FREEBSD_DMA_ALLOCATION_SLOT*
_NalFindDmaAllocationSlot(
        IN  KVOID*                              KernelAddress
        );

KVOID*
_NalAllocateDmaMemory(
        IN  bus_dma_tag_t                       Parent ,
        IN  NAL_FREEBSD_DMA_ALLOCATION_SLOT*    DmaAllocationSlot ,
        IN  UINT32                              ByteCount ,
        IN  UINT32                              Alignment ,
        OUT NAL_PHYSICAL_ADDRESS*               PhysicalAddress,
        IN  CHAR*                               NamedLocator,
        IN  UINT32                              LineNumber
        );
VOID
_NalFreeDmaMemory(
        IN  NAL_FREEBSD_DMA_ALLOCATION_SLOT*    DmaAllocationSlot
        );

NAL_FREEBSD_PCI_ALLOCATION_SLOT*
_NalFindPciResourcesAllocationSlot(
        IN  KVOID*                              KernelAddress
        );

KVOID*
_NalAllocatePciResources(
        IN  device_t                            Device ,
        IN  NAL_FREEBSD_PCI_ALLOCATION_SLOT*    PciAllocationSlot ,
        IN  UINT32                              ByteCount ,
        IN  UINT32                              Alignment ,
        OUT NAL_PHYSICAL_ADDRESS*               PhysicalAddress,
        IN  CHAR*                               NamedLocator,
        IN  UINT32                              LineNumber
        );
VOID
_NalFreePciResources(
        IN  NAL_FREEBSD_PCI_ALLOCATION_SLOT*    PciAllocationSlot
        );

/***************************************************************************
**
** Name:            NalMmapAddress()
**
** Description:     Maps a physical address to a virtual address for I/O.
**
** Arguments:       VirtualAddress  = Pointer to the Virtual Address to map to
**                  PhysicalAddress = The address to map
**                  Length          = Number of bytes to map
**                  ProcessId       = id of process
**
** Returns:         NAL_STATUS
**                  Length contains the number of bytes actually mapped.
**                  VirtualAddress contains the base address of the mapped region
**
****************************************************************************/
NAL_STATUS
NalMmapAddressEx(
    IN OUT  KVOID**                 VirtualAddress,
    IN      NAL_PHYSICAL_ADDRESS    PhysicalAddress,
    IN OUT  UINT32*                 Length,
    IN      UINTN                   ProcessId
    )
{
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            NalUnmapAddress()
**
** Description:     Unmaps an address mapped with NalMmapAddress().
**
** Arguments:       VirtualAddress  = Pointer to the Virtual Address to map to
**                  PhysicalAddress = The physical address mapped to the
**                                    VirtualAddress parameter.
**                  Length          = Number of bytes mapped.
**
** Returns:         NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
NalUnmapAddressEx(
    IN  KVOID*                  VirtualAddress,
    IN  NAL_PHYSICAL_ADDRESS    PhysicalAddress,
    IN  UINT32                  Length,
    IN  UINTN                   ProcessId
    )
{
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            _NalAllocateMemoryNonPagedEx()
**
** Description:     The function allocates memory non paged.
**
** Arguments:       ByteCount           = Number of bytes to allocate
**                  Alignment           = Alignment of bytes
**                  ProcessId           = Id of current process
**                  PhysicalAddress     = Pointer to the physical address
**                                        destination or NULL if not used
**                  NamedLocator        = name of the file that allocates memory
**                  LineNumber          = Line of allocation
**
** Returns:         Allocated address
**
****************************************************************************/
KVOID*
_NalAllocateMemoryNonPagedEx(
    IN  UINT32                  ByteCount,
    IN  UINT32                  Alignment,
    IN  UINTN                   ProcessId,
    OUT NAL_PHYSICAL_ADDRESS*   PhysicalAddress, /* OPTIONAL */
    IN  CHAR*                   NamedLocator,
    IN  UINT32                  LineNumber
    )
{
    return _NalAllocateMemoryNonPaged(ByteCount,Alignment,PhysicalAddress,NamedLocator,LineNumber);
}

/***************************************************************************
**
** Name:            _NalFreeMemoryNonPagedEx()
**
** Description:     Releases non paged memory
**
** Arguments:       Address     = Allocated address to release
**                  ProcessId   = Id of current process
**                  NamedLocator        = name of the file that releases memory
**                  LineNumber          = Line of releases
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalFreeMemoryNonPagedEx(
    IN  KVOID*      Address,
    IN  UINTN       ProcessId,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{
    _NalFreeMemoryNonPaged(Address,NamedLocator,LineNumber);
}

/***************************************************************************
**
** Name:            NalGetPhysicalMemoryAddressEx()
**
** Description:
**
** Arguments:
**
** Returns:
**
****************************************************************************/
NAL_PHYSICAL_ADDRESS
NalGetPhysicalMemoryAddressEx(
    IN  KVOID*                  VirtualAddress,
    IN  UINTN                   ProcessId
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalEnableDebugPrint()
**
** Description:
** Arguments:
** Returns:
**
****************************************************************************/
VOID
NalEnableDebugPrint(
    IN BOOLEAN      Enable
    )
{

}

/***************************************************************************
**
** Name:            NalDebugPrint()
**
** Description:
** Arguments:
** Returns:
**
****************************************************************************/
NAL_STATUS
NalDebugPrint(
    IN CHAR* Format,
    ...
    )
{
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            NalDebugPrintSupport()
**
** Description:
** Arguments:
** Returns:
**
****************************************************************************/
NAL_STATUS
NalDebugPrintSupport(
    OUT UINT32* DebugCapabilities
    )
{
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            NalKMemSet()
**
** Description:     Sets Dest to the value of Value for Size length.
**
** Arguments:       Dest     = Pointer to the address to set
**                  Value    = Integer value to set.
**                  Size     = Bytes to set
**
** Returns:         Dest
**
****************************************************************************/
KVOID*
NalKMemset(
    IN      KVOID*      Dest,
    IN      int         Value,
    IN      UINTN       Size
    )
{
    return memset(Dest, Value, Size);
}

/***************************************************************************
**
** Name:            NalKtoUMemcpy()
**
** Description:     Copies a buffer from Kernel to User space of Size bytes.
**
** Arguments:       Source     = Pointer to the Kernel Address.
**                  Dest       = Pointer to the User Address.
**                  Size       = Bytes to Copy
**
** Returns:         Dest, or NULL if there was an error
**
****************************************************************************/
VOID*
NalKtoUMemcpy(
    IN      VOID*       Dest,
    IN      KVOID*      Source,
    IN      UINTN       Size
    )
{
    VOID * ResultPointer = NULL;
    int    ErrorCode     = 0;

    ErrorCode = copyout(Source , Dest , Size);

    if(ErrorCode == 0)
    {
        ResultPointer = Dest;
    }

    return ResultPointer;
}

/***************************************************************************
**
** Name:            NalKtoKMemcpy()
**
** Description:     Does a kernel mode memcpy between two kernel mode buffers.
**
** Arguments:       Source     = Pointer to the Kernel Address.
**                  Dest       = Pointer to the Kenerl Address.
**                  Size       = Bytes to Copy
**
** Returns:         Dest
**
****************************************************************************/
KVOID*
NalKtoKMemcpy(
    IN      KVOID*      Dest,
    IN      KVOID*      Source,
    IN      UINTN       Size
    )
{
    return memcpy(Dest, Source, Size);
}

/***************************************************************************
**
** Name:            NalUtoKMemcpy()
**
** Description:     Copies a buffer from User to Kernel space of Size bytes.
**
** Arguments:       Source     = Pointer to the User Address.
**                  Dest       = Pointer to the Kernel Address.
**                  Size       = Bytes to Copy
**
** Returns:         Dest or NULL if error
**
****************************************************************************/
KVOID*
NalUtoKMemcpy(
    IN      KVOID*      Dest,
    IN      VOID*       Source,
    IN      UINTN       Size
    )
{
    VOID * ResultPointer = NULL;
    int    ErrorCode     = 0;

    ErrorCode = copyin(Source , Dest , Size);

    if(ErrorCode == 0)
    {
        ResultPointer = Dest;
    }

    return ResultPointer;
}

/***************************************************************************
**
** Name:            NalReadPort8()
**
** Description:     Reads an 8 bit value from a specified I/O Port
**
** Arguments:       Port = the I/O port from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT8
NalReadPort8(
    IN  PORT_ADDR Port
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalReadPort16()
**
** Description:     Reads a 16bit value from a specified I/O Port
**
** Arguments:       Port = the I/O port from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT16
NalReadPort16(
    IN  PORT_ADDR Port
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalReadPort32()
**
** Description:     Reads a 32bit value from a specified I/O Port
**
** Arguments:       Port = the I/O port from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT32
NalReadPort32(
    IN  PORT_ADDR Port
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalWritePort8()
**
** Description:     Writes an 8bit value to a specified I/O Port
**
** Arguments:       Port  = the I/O port to which to write.
**                  Value = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWritePort8(
    IN  PORT_ADDR   Port,
    IN  UINT8       Value
    )
{
    return FALSE;
}

/***************************************************************************
**
** Name:            NalWritePort16()
**
** Description:     Writes a 16bit value to a specified I/O Port
**
** Arguments:       Port  = the I/O port to which to write.
**                  Value = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWritePort16(
    IN  PORT_ADDR   Port,
    IN  UINT16      Value
    )
{
    return FALSE;
}

/***************************************************************************
**
** Name:            NalWritePort32()
**
** Description:     Writes a 32 bit value to a specified I/O Port
**
** Arguments:       Port  = the I/O port to which to write.
**                  Value = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWritePort32(
    IN  PORT_ADDR   Port,
    IN  UINT32      Value
    )
{
    return FALSE;
}

/***************************************************************************
**
** Name:            NalReadRegister8()
**
** Description:     Reads an 8 bit value from the specified memory address
**
** Arguments:       Address  = the memory address from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT8
NalReadRegister8(
    IN  KVOID*   Address
    )
{
    UINT8   Value = 0;

    return Value;
}

/***************************************************************************
**
** Name:            NalReadRegister16()
**
** Description:     Reads a 16 bit value from the specified memory address
**
** Arguments:       Address  = the memory address from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT16
NalReadRegister16(
    IN  KVOID*    Address
    )
{
    UINT16   Value = 0;

    return Value;
}

/***************************************************************************
**
** Name:            NalReadRegister32()
**
** Description:     Reads a 32bit value from the specified memory address
**
** Arguments:       Address  = the memory address from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT32
NalReadRegister32(
    IN  KVOID*        Address
    )
{
    UINT32   Value = 0;

    return Value;
}

/***************************************************************************
**
** Name:            NalWriteRegister8()
**
** Description:     Writes an 8bit value to a specified Memory Address.
**
** Arguments:       Address  = Pointer to the address to write to.
**                  Value    = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWriteRegister8(
    IN  KVOID*                   Address,
    IN  UINT8                    Value
    )
{
    BOOLEAN Success = FALSE;

    return Success;
}

/***************************************************************************
**
** Name:            NalWriteRegister16()
**
** Description:     Writes a 16bit value to a specified Memory Address.
**
** Arguments:       Address  = Pointer to the address to write to.
**                  Value    = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWriteRegister16(
    IN  KVOID*                Address,
    IN  UINT16                Value
    )
{
    BOOLEAN Success = FALSE;

    return Success;
}

/***************************************************************************
**
** Name:            NalWriteRegister32()
**
** Description:     Writes a 32bit value to a specified Memory Address.
**
** Arguments:       Address  = Pointer to the address to write to.
**                  Value    = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWriteRegister32(
    IN  KVOID*                   Address,
    IN  UINT32                   Value
    )
{
    BOOLEAN Success = FALSE;

    return Success;
}

/***************************************************************************
**
** Name:            NalMmapAddress()
**
** Description:     Maps a physical address to a virtual address for I/O.
**
** Arguments:       VirtualAddress  = Pointer to the Virtual Address to map to
**                  PhysicalAddress = The address to map
**                  Length          = Number of bytes to map
**
** Returns:         NAL_STATUS
**                  Length contains the number of bytes actually mapped.
**                  VirtualAddress contains the base address of the mapped region
**
****************************************************************************/
NAL_STATUS
NalMmapAddress(
    IN OUT  KVOID**                 VirtualAddress,
    IN      NAL_PHYSICAL_ADDRESS    PhysicalAddress,
    IN OUT  UINT32*                 Length
    )
{
    NAL_STATUS  NalStatus = NAL_NOT_IMPLEMENTED;
    return NalStatus;
}

/***************************************************************************
**
** Name:            NalUnmapAddress()
**
** Description:     Unmaps an address mapped with NalMmapAddress().
**
** Arguments:       VirtualAddress  = Pointer to the Virtual Address to map to
**                  PhysicalAddress = The physical address mapped to the
**                                    VirtualAddress parameter.
**                  Length          = Number of bytes mapped.
**
** Returns:         NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
NalUnmapAddress(
    IN  KVOID*                  VirtualAddress,
    IN  NAL_PHYSICAL_ADDRESS    PhysicalAddress,
    IN  UINT32                  Length
    )
{
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            NalGetTimeStamp()
**
** Description:     Returns the TSC (Time Stamp Counter - IA32) or the ITC
**                  (Interval Time Counter - IA64).
**
** Arguments:       NONE
**
** Returns:         UINT64 - 64-bit time stamp that represents the number of
**                      clock increments since the CPU has started.
**
****************************************************************************/
UINT64
NalGetTimeStamp(VOID)
{
    return 0;
}

/***************************************************************************
**
** Name:            NalGetTimeStampsPerMicrosecond()
**
** Description:     Returns the number of time stamps in each microsecond.
**
** Arguments:       NONE
**
** Returns:         UINT64 - The number of time stamps that will occur during
**                           a microsecond.
**
** Note:            This is also implemented in user-mode of the FReeBSD OS Interface.
**                  It is only included here because the NAL IOCTL interface requires it.
****************************************************************************/
UINT64
NalGetTimeStampsPerMicrosecond(VOID)
{
    return (UINT64)0;
}


/***************************************************************************
**
** Name:            _NalAllocateMemory()
**
** Description:     Allocates the specified number of pagable bytes.
**
** Arguments:       ByteCount  = Number of bytes to allocate.
**                  NamedLocator = Name such as File name, __FILE__ works.
**                  LineNumber = Line Number in the code file. __LINE__ works.
**
** Returns:         Pointer to the memory buffer, NULL if failed.
**
** Note:            This is used ONLY to allocate kernel memory from functions within
**                  the kernel portion of the NAL.  No IOCTL call will resolve to this.
**                  Instead, they will use malloc.
****************************************************************************/
VOID*
_NalAllocateMemory(
    IN  UINT32      ByteCount,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            _NalFreeMemory()
**
** Description:     Frees the memory allocated at the buffer address.
**
** Arguments:       Address  = Pointer to the address to free.
**                  NamedLocator = Name such as File name, __FILE__ works.
**                  LineNumber = Line Number in the code file. __LINE__ works.
**
** Returns:         Nothing.
**
** Note:            This is used ONLY to free kernel memory allocated from within
**                  the kernel portion of the NAL.  No IOCTL call will resolve to this.
****************************************************************************/
VOID
_NalFreeMemory(
    IN  VOID*       Address,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{

}

/***************************************************************************
**
** Name:            NalDelayMilliseconds()
**
** Description:     Stops the thread for a specified amt of milliseconds
**
** Arguments:       Milliseconds = Number of milliseconds to pause.
**
** Returns:         Nothing.
**
****************************************************************************/
VOID
NalDelayMilliseconds(
    IN  UINT32 Milliseconds
    )
{
}

/***************************************************************************
**
** Name:            NalDelayMicroseconds()
**
** Description:     Stops the thread for a specified amt of microseconds
**
** Arguments:       Microseconds = Number of microseconds to pause.
**
** Returns:         Nothing.
**
****************************************************************************/
VOID
NalDelayMicroseconds(
    IN  UINT32 Microseconds
    )
{

}

/***************************************************************************
**
** Name:            NalGetPhysicalMemoryAddress()
**
** Description:     Returns the Physical address associated with this virtual
**                  address. This address can be used for DMA by HW.
**
** Arguments:       Address  = Pointer to the address to resolve.
**
** Returns:         Physical Address.
**
** Note:            EFI maps physical addresses to their virtual addresses,
**                  so this returns the virtual address casted.
**
****************************************************************************/
NAL_PHYSICAL_ADDRESS
NalGetPhysicalMemoryAddress(
    IN  KVOID* VirtualAddress
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            _NalFreebsdDmaLoadCallback()
**
** Description:     Callback for DMA memory allocations
**
** Arguments:       Argument            - Non pageed pci allocation slot element
**                  Segment             - Allocated segment
**                  NumberOfSegments    - Number of allocated segments
**                  Error               - code of error or 0 if success
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalFreebsdDmaLoadCallback(VOID *Argument, bus_dma_segment_t * Segment, int NumberOfSegments, int Error)
{
    NAL_FREEBSD_PCI_ALLOCATION_SLOT * NonPagedElement = (NAL_FREEBSD_PCI_ALLOCATION_SLOT *) Argument;
    if(Error == 0)
    {
        NonPagedElement->PhysicalAddress = Segment->ds_addr;
    }
    else
    {
        NonPagedElement->PhysicalAddress = PHYSICAL_NULL;
    }
}

/***************************************************************************
**
** Name:            _NalFindDmaAllocationSlot()
**
** Description:     Searching the allocation slot in the global array.
**
** Arguments:       KernelAddress       - Try to find a slot, that is
**                                        assigned for this address.
**                                        If it will be set to NULL,
**                                        then the function will
**                                        search empty slot
**
** Returns:         nothing
**
****************************************************************************/
NAL_FREEBSD_DMA_ALLOCATION_SLOT*
_NalFindDmaAllocationSlot(
        IN  KVOID*                  KernelAddress
        )
{
    NAL_FREEBSD_DMA_ALLOCATION_SLOT*    DmaAllocationSlot = NULL;
    UINT32                              Index             = 0;

    for(Index=0; Index<NAL_FREEBSD_MAX_NON_PAGED_MEMORY_ALLOCATIONS ;Index++)
    {
        if(Global_DmaAllocationSlotsTable[Index].KernelAddress == KernelAddress)
        {
            DmaAllocationSlot = &Global_DmaAllocationSlotsTable[Index];
            break;
        }
    }


    return DmaAllocationSlot;
}

/***************************************************************************
**
** Name:            _NalAllocateDmaMemory()
**
** Description:     Allocates DMA memory
**
** Arguments:       Parent              - parent tag (for example for the device)
**                  DmaAllocationSlot   - allocation slot from the _NalFindDmaAllocationSlot
**                                        function.
**                  ByteCount           - Number of bytes to allocate
**                  Alignment           - Alignment of bytes
**                  PhysicalAddress     - Pointer to the destination of the physical address
**                                        of allocated memory, or NULL if not used
**                  NamedAllocator      - Name of the file, that allocate memory
**                  LineNumber          - number of line in the file, that allocates
**                                        this memory
**
** Returns:         allocated DMA memory address
**
****************************************************************************/
KVOID*
_NalAllocateDmaMemory(
        IN  bus_dma_tag_t                        Parent ,
        IN  NAL_FREEBSD_DMA_ALLOCATION_SLOT*     DmaAllocationSlot ,
        IN  UINT32                               ByteCount ,
        IN  UINT32                               Alignment ,
        OUT NAL_PHYSICAL_ADDRESS*                PhysicalAddress,
        IN  CHAR*                                NamedLocator,
        IN  UINT32                               LineNumber
        )
{
    KVOID* KernelAddress = NULL;
    int    Result        = -1;

    DmaAllocationSlot->PhysicalAddress  = PHYSICAL_NULL;

    if(Alignment == 0)
    {
        Alignment = 1;
    }
    do
    {
        Result = bus_dma_tag_create(Parent,
                                    Alignment,
                                    0,
                                    BUS_SPACE_MAXADDR,
                                    BUS_SPACE_MAXADDR,
                                    NULL,
                                    NULL,
                                    ByteCount,
                                    1,
                                    ByteCount,
                                    BUS_DMA_ALLOCNOW,
                                    NULL,
                                    NULL,
                                    &DmaAllocationSlot->DmaTag);

        if(Result != 0)
        {
            NalDebugPrint("_NalAllocateDmaMemory: Cannot create tag for allocation request from %s:%d\n",NamedLocator,LineNumber);
            NalDebugPrint("                       Function return %d\n",Result);
            break;
        }

        Result = bus_dmamem_alloc(DmaAllocationSlot->DmaTag,
                                  (void **)&KernelAddress,
                                  BUS_DMA_NOWAIT,
                                  &DmaAllocationSlot->DmaMap);

        if(Result != 0)
        {
            NalDebugPrint("_NalAllocateDmaMemory: Cannot create DMA memory for allocation request from %s:%d\n",NamedLocator,LineNumber);
            NalDebugPrint("                       Function return %d\n",Result);
            KernelAddress = NULL;
            break;
        }

        Result = bus_dmamap_load(DmaAllocationSlot->DmaTag,
                                 DmaAllocationSlot->DmaMap,
                                 KernelAddress,
                                 ByteCount,
                                 _NalFreebsdDmaLoadCallback,
                                 DmaAllocationSlot,
                                 BUS_DMA_NOWAIT);

        if(Result != 0 || DmaAllocationSlot->PhysicalAddress == PHYSICAL_NULL)
        {
            NalDebugPrint("_NalAllocateDmaMemory: Cannot create a mapping in device visible address space for allocation request from %s:%d\n",NamedLocator,LineNumber);
            NalDebugPrint("                       Function return %d\n",Result);
            KernelAddress = NULL;
        }
    } while(0);

    if(KernelAddress == NULL || DmaAllocationSlot->PhysicalAddress == PHYSICAL_NULL)
    {
        _NalFreeDmaMemory(DmaAllocationSlot);
    }
    else
    {
        DmaAllocationSlot->KernelAddress = KernelAddress;
        if(PhysicalAddress != NULL)
        {
            *PhysicalAddress = DmaAllocationSlot->PhysicalAddress;
        }
    }

    return KernelAddress;
}

/***************************************************************************
**
** Name:            _NalFreeDmaMemory()
**
** Description:     Releases DMA memory
**
** Arguments:       DmaAllocationSlot       - Allocated slot from the
**                                           _NalFindDmaAllocationSlot function.
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalFreeDmaMemory(
        IN  NAL_FREEBSD_DMA_ALLOCATION_SLOT*     DmaAllocationSlot
        )
{
    bus_dmamap_unload(DmaAllocationSlot->DmaTag, DmaAllocationSlot->DmaMap);
    bus_dmamem_free(DmaAllocationSlot->DmaTag, DmaAllocationSlot->KernelAddress , DmaAllocationSlot->DmaMap);
    bus_dma_tag_destroy(DmaAllocationSlot->DmaTag);
    memset(DmaAllocationSlot, 0, sizeof(NAL_FREEBSD_DMA_ALLOCATION_SLOT));
}

/***************************************************************************
**
** Name:            _NalFindPciAllocationSlot()
**
** Description:     Searching a PCI resources allocation slot
**
** Arguments:       KernelAddress       - Try to find a slot, that is
**                                        assigned for this address.
**                                        If it will be set to NULL,
**                                        then the function will
**                                        search empty slot
**
** Returns:         Slot of PCI memory
**
****************************************************************************/
NAL_FREEBSD_PCI_ALLOCATION_SLOT*
_NalFindPciResourcesAllocationSlot(
        IN  KVOID*                  KernelAddress
        )
{
    NAL_FREEBSD_PCI_ALLOCATION_SLOT*    PciAllocationSlot = NULL;
    UINT32                              Index             = 0;

    for(Index=0; Index<NAL_FREEBSD_MAX_NON_PAGED_MEMORY_ALLOCATIONS ;Index++)
    {
        if(Global_PciAllocationSlotsTable[Index].KernelAddress == KernelAddress)
        {
            PciAllocationSlot = &Global_PciAllocationSlotsTable[Index];
            break;
        }
    }


    return PciAllocationSlot;
}

/***************************************************************************
**
** Name:            _NalAllocatePciResources()
**
** Description:     Allocate resources for the PCI device
**
** Arguments:       Device              - PCI device for the allocation
**                  PciAllocationSlot   - Slot for allocation from the
**                                        _NalFindPciAllocationSlot
**                  ByteCount           - Number of bytes to allocate
**                  Alignment           - Alignment of memory
**                  PhysicalAddress     - Pointer to the destination of the physical address
**                                        of allocated memory, or NULL if not used
**                  NamedAllocator      - Name of the file, that allocate memory
**                  LineNumber          - number of line in the file, that allocates
**                                        this memory
**
** Returns:         allocated PCI memory address
**
****************************************************************************/
KVOID*
_NalAllocatePciResources(
        IN  device_t                             Device ,
        IN  NAL_FREEBSD_PCI_ALLOCATION_SLOT*     PciAllocationSlot ,
        IN  UINT32                               ByteCount ,
        IN  UINT32                               Alignment ,
        OUT NAL_PHYSICAL_ADDRESS*                PhysicalAddress,
        IN  CHAR*                                NamedLocator,
        IN  UINT32                               LineNumber
        )
{
    PciAllocationSlot->Device           = Device;
    PciAllocationSlot->RegisterOffset   = PCIR_BAR(0);

    PciAllocationSlot->Resource         = bus_alloc_resource(Device, SYS_RES_MEMORY, &PciAllocationSlot->RegisterOffset, 0ul,~0ul,ByteCount, RF_ACTIVE | RF_SHAREABLE );

    if(PciAllocationSlot->Resource)
    {
        PciAllocationSlot->KernelAddress    = rman_get_virtual(PciAllocationSlot->Resource);
        PciAllocationSlot->PhysicalAddress  = rman_get_bushandle(PciAllocationSlot->Resource);

        if(PhysicalAddress != NULL)
        {
            *PhysicalAddress = PciAllocationSlot->PhysicalAddress;
        }
    }

    return PciAllocationSlot->KernelAddress;
}

/***************************************************************************
**
** Name:            _NalFreePciResources()
**
** Description:     Releases PCI resources
**
** Arguments:       DmaAllocationSlot       - Allocated slot from the
**                                           _NalFindPciAllocationSlot function.
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalFreePciResources(
        IN  NAL_FREEBSD_PCI_ALLOCATION_SLOT*     PciAllocationSlot
        )
{
    bus_release_resource(PciAllocationSlot->Device,SYS_RES_MEMORY,PciAllocationSlot->RegisterOffset,PciAllocationSlot->Resource);
    memset(PciAllocationSlot, 0, sizeof(NAL_FREEBSD_PCI_ALLOCATION_SLOT));
}

/***************************************************************************
**
** Name:            _NalAllocateMemoryNonPaged()
**
** Description:     Allocates the specified number of bytes. These bytes are contiguous
**                  non-paged.
**
** Arguments:       ByteCount  = Number of bytes to allocate.
**                  Alignment = 32bit alignment requirements such as 4096.
**                  PhysicalAddress = Will contain the physical address of allocated
**                                    memory block. If this call fails, this will
**                                    be set to NULL. It is optional, so if it is
**                                    NULL, it will be ignored.
**                  NamedLocator = ASCII string that can identify filename or function
**                                 name where memory allocation is performed.
**                  LineNumber = The line number in NamedLocator where memory's allocated
**
** Returns:         Pointer to the memory buffer, NULL if failed.
**
****************************************************************************/
KVOID*
_NalAllocateMemoryNonPaged(
    IN  UINT32                  ByteCount,
    IN  UINT32                  Alignment,
    OUT NAL_PHYSICAL_ADDRESS*   PhysicalAddress, /* OPTIONAL */
    IN  CHAR*                   NamedLocator,
    IN  UINT32                  LineNumber
    )
{
    KVOID *                           KernelAddress     = NULL;
    NAL_FREEBSD_DMA_ALLOCATION_SLOT * DmaAllocationSlot = NULL;

    DmaAllocationSlot = _NalFindDmaAllocationSlot(NULL);

    if(DmaAllocationSlot != NULL)
    {
        KernelAddress = _NalAllocateDmaMemory(NULL,
                              DmaAllocationSlot,
                              ByteCount,
                              Alignment,
                              PhysicalAddress,
                              NamedLocator,
                              LineNumber);
    }

    return KernelAddress;
}

/***************************************************************************
**
** Name:            _NalFreeMemoryNonPaged()
**
** Description:     Frees the memory allocated at the buffer address.
**
** Arguments:       Address  = Pointer to the address to free.
**
** Returns:         Nothing.
**
****************************************************************************/
VOID
_NalFreeMemoryNonPaged(
    IN  KVOID*      Address,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{
    NAL_FREEBSD_DMA_ALLOCATION_SLOT * DmaAllocationSlot  = NULL;

    DmaAllocationSlot = _NalFindDmaAllocationSlot(Address);

    if(DmaAllocationSlot != NULL)
    {
        _NalFreeDmaMemory(DmaAllocationSlot);
    }
}

/***************************************************************************
**
** Name:            _NalAllocateMemoryNonPagedPci()
**
** Description:     Allocates the specified number of bytes of DMA memory for PCI
**                  device. Note, that it is not allocate PCI Resources
**
** Arguments:       PDev      = Pointer to the device
**                  ByteCount = Number of bytes to allocate.
**                  Alignment = 32bit alignment requirements such as 4096.
**                  PhysicalAddress = Will contain the physical address of allocated
**                                    memory block. If this call fails, this will
**                                    be set to NULL. It is optional, so if it is
**                                    NULL, it will be ignored.
**                  NamedLocator = ASCII string that can identify filename or function
**                                 name where memory allocation is performed.
**                  LineNumber = The line number in NamedLocator where memory's allocated
**
** Returns:         Pointer to the memory buffer, NULL if failed.
**
****************************************************************************/
KVOID*
_NalAllocateMemoryNonPagedPci(
    IN  KVOID*                  PDev,
    IN  UINT32                  ByteCount,
    IN  UINT32                  Alignment,
    OUT NAL_PHYSICAL_ADDRESS*   PhysicalAddress,
    IN  CHAR*                   NamedLocator,
    IN  UINT32                  LineNumber
    )
{
    KVOID *                           KernelAddress      = NULL;
    NAL_FREEBSD_DMA_ALLOCATION_SLOT * DmaAllocationSlot  = NULL;
    device_t                          Device             = (device_t)PDev;

    DmaAllocationSlot = _NalFindDmaAllocationSlot(KernelAddress);

    if(DmaAllocationSlot != NULL)
    {

        KernelAddress = _NalAllocateDmaMemory(bus_get_dma_tag(Device) ,
                                              DmaAllocationSlot,
                                              ByteCount,
                                              Alignment,
                                              PhysicalAddress,
                                              NamedLocator,
                                              LineNumber);
    }

    return KernelAddress;
}

/***************************************************************************
**
** Name:            _NalFreeMemoryNonPagedPci()
**
** Description:     Releases memory allocated for the PCI device.
**
** Arguments:       PDev        = Pointer to the device
**                  Address     = Allocated address
**
** Returns:         Nothing
**
****************************************************************************/
VOID
_NalFreeMemoryNonPagedPci(
    IN  KVOID*  PDev,
    IN  KVOID*  Address
    )
{
    NAL_FREEBSD_DMA_ALLOCATION_SLOT * DmaAllocationSlot  = NULL;

    DmaAllocationSlot = _NalFindDmaAllocationSlot(Address);

    if(DmaAllocationSlot != NULL)
    {
        _NalFreeDmaMemory(DmaAllocationSlot);
    }
}

/***************************************************************************
**
** Name:            NalAtomicIncrement32()
**
** Description:     Atomically increments a UINT32 Pointer.
**
** Arguments:       Address  = Pointer to the address to increment
**
** Returns:         Value at incremented Address
**
** Note:            This assumes that the Address value is a user-mode value. This function cannot
**                  be called from within this driver to work on a kernel address. It's using
**                  copy_to_user/copy_from_user
****************************************************************************/
UINT32
NalAtomicIncrement32(
    IN  UINT32* Address
    )
{
    atomic_add_32(Address , 1);
    return *Address;
}

/***************************************************************************
**
** Name:            NalAtomicDecrement32()
**
** Description:     Atomically decrements a UINT32 Pointer.
**
** Arguments:       Address  = Pointer to the address to decrement
**
** Returns:         Value at decremented Address
**
** Note:            This assumes that the Address value is a user-mode value. This function cannot
**                  be called from within this driver to work on a kernel address. It's using
**                  copy_to_user/copy_from_user
****************************************************************************/
UINT32
NalAtomicDecrement32(
    IN  UINT32* Address
    )
{
    atomic_subtract_32(Address,1);
    return *Address;
}

/***************************************************************************
**
** Name:            NalAtomicTestSet32()
**
** Description:     If "Address" contains the same value as "Test", this will
**                  set "Address" to contain "Set" instead.
**
** Arguments:       Address  = Pointer to the address to test & set
**                  Test     = 32bit value to test
**                  Set      = 32bit value to set
**
** Returns:         Value at prior to being modified Address
**
** Note:            This assumes that the Address value is a user-mode value. This function cannot
**                  be called from within this driver to work on a kernel address. It's using
**                  copy_to_user/copy_from_user
**
****************************************************************************/
UINT32
NalAtomicTestSet32(
    IN  UINT32* Address,
    IN  UINT32  Test,
    IN  UINT32  Set
    )
{
    UINT32  ReturnValue  = 0;
    UINT32  AddressLocal = 0;

    if(Address != NULL)
    {
        /* FreeBSD doesnt provide this API, so we use a SpinLock */
        mtx_lock(&Global_AtomicTestSetSpinLock);

        copyin(Address , &AddressLocal , sizeof(UINT32));

        /* Return *Address regardless if changed or not. */
        ReturnValue = AddressLocal;

        /* Test now, then set if true. */
        if(AddressLocal == Test)
        {
            AddressLocal = Set;
        }

        copyout(&AddressLocal , Address , sizeof(UINT32));

        /* Release the spinlock */
        mtx_unlock(&Global_AtomicTestSetSpinLock);
    }


    return ReturnValue;
}
